function d=get(data,label)
%GET Access/query dataset property values
%    
%    D=GET(DATA,LABEL) returns the value of the specified 
%    property LABEL of the data set D. Possibles properties
%    are:
%    
%      NV: the number of variables
%      NS: the number of examples
%      NC: the number of classes
%      CONTDATA: the continuous data
%      SYMBDATA: the symbolic data
%      CLASSES: the classes
%      CLASSESLIST: the list of classes
%      DATA: the data attribute
%      SYMBOLS: the symbols attribute
%      VARTYPES: the vartypes attribute
%      LABELS: the labels attribute
%      VARIABLES: the labels attribute
%    
%    See also: SET 
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose	Access/query dataset property values.
%
%   @synopsis	d=get(data,label)
%   @description  returns the value of the specified 
%   property <code>label</code> of the data set <code>d</code>.
%	Possibles properties are:
%  	<dl>
%	<dt> nv <dd> the number of variables
%	<dt> ns <dd> the number of examples
%	<dt> nc <dd> the number of classes
%	<dt> contdata <dd> the continuous data
%	<dt> symbdata <dd> the symbolic data
%	<dt> classes <dd> the classes
%	<dt> classeslist <dd> the list of classes
%	<dt> data <dd> the data attribute
%	<dt> symbols <dd> the symbols attribute
%	<dt> vartypes <dd> the vartypes attribute
%	<dt> labels <dd> the labels attribute
%	<dt> variables <dd> the labels attribute
%	</dl>
%
%   @see set 
%
%MAN_PAGE_END

% By Antoine Duchateau. All Rights Reserved.
% Written: 19/03/99

if nargin == 2,
   
   switch upper(label),
      
   case 'NV',												%Number of features
      d = data.nv;
      
   case 'CLASSESDISTRI'
      nc = get(data,'nc');
      classes = get(data,'classes');
      
      d = zeros(1,nc);
      for i=1:nc,
         d(i) = sum(classes == i);
      end
      
   case 'CONTDATA',										%continuous data
      d = get_data(data,':',find(strcmp(data.vartypes,'continuous')));
      
   case 'SYMBDATA',										%symbolic data
      d = get_data(data,':',find(strcmp(data.vartypes,'symbolic')));
      
   case 'DATA',										%symbolic data
      d = get_data(data,':',find(~strcmp(data.vartypes,'class')));
   
   case 'CLASSES',											%classes
      d = get_data(data,':',find(strcmp(data.vartypes,'class')));
      
   case 'CLASSESLIST',											%classes
      d = data.symbols{find(strcmp(data.vartypes,'class'))};
      
   case 'NS',												%Number of examples
      d = data.ns;
      
   case 'NC',
      d = length(data.symbols{find(strcmp(data.vartypes,'class'))});
      
   otherwise
      d = getfield(struct(data),lower(label));
   end
elseif nargin == 1,
   help dataset/get;
end

