function m = set(m,varargin)
%SET Set object properties
%    
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M.
%    
%    M_OUT =
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M and return the modified model
%    inside m_out. The original model is not modified. The
%    following ATTRNAME are recognised:
%    
%      NAME: sets the name of the system
%      N_IN: sets the number of inputs of the system
%      N_OUT: sets the number of outputs of the system
%      DATA: sets the data set linked to the system
%      T: sets the sampling period of the discrete system, 0
%          if continuous
%      MAPPING: sets the array of mappings
%      LINKS: sets the links array which defines which
%          output is computed by which mapping
%      LIMITS: sets the limits of the domain of definition
%          of the mapping. The value must be a 2*N_IN matrix
%          where the upper and lever saturation level are
%      USERDATA: sets the 'userData' field
%      OPT: sets the 'opt' field
%      DATE: sets the date of creation of the object
%      MAPFIELD: let the user access directly property of
%          the mappings. When used, VALUE must contain a 3
%          elements cell array {N 'property' MAPVALUE}. N is
%          the index of the	mapping, 'property' is the name
%          of the property which has to be changet and
%          MAPVALUE is the new value. An even more
%          convenient way to change mappings' fields is a
%          achieved through the syntax s{n}.property =
%          mapvalue.
%      PARAMS: sets the parameters to be optimised
%      DYNAMICS: sets the dynamics of the system (please use
%          add_dynamics instead)
%    
%    See also: GET
%    
%    

%MAN_PAGE_BEGIN
%
%  @purpose Set object properties.
%  
%  @synopsis set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%  @description sets the value of the attribute <code>AttrName1</code>,
%  <code>AttrName2</code>, ... associated to the object <code>m</code>.
%
%  @synopsis  m_out = set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%  @description sets the value of the attribute <code>AttrName1</code>,
%  <code>AttrName2</code>, ... associated to the object <code>m</code> and
%  return the modified model inside m_out. The original model is not modified.
%  The following <CODE>AttrName</CODE> are recognised:
%
%<dl>
%<dt>		name  <dd>	sets the name of the system
%<dt>     n_in  <dd>      	sets the number of inputs of the system
%<dt>     n_out  <dd>     	sets the number of outputs of the system
%<dt>			data <dd>		sets the data set linked to the system
%<dt>			T <dd>		sets the sampling period of the discrete system, 0 if continuous
%<dt>			mapping <dd>		sets the array of mappings
%<dt>			links <dd>			sets the links array which defines which output is computed by which mapping
%<dt>	 limits	 <dd>				sets the limits of the domain of definition
%								of the mapping. The value must be a <code>2*n_in</code> matrix where
%													the upper and lever saturation level are
%<dt>			userData <dd>		sets the 'userData' field
%<dt>			opt			 <dd>		sets the 'opt' field
%<dt>			date		 <dd>		sets the date of creation of the object
%<dt>	 mapfield	 <dd>	let the user access directly property of the mappings. When used, <CODE>value</CODE>
%must contain a 3 elements cell array {<code>n</code> 'property' <CODE>mapvalue</CODE>}. <code>n</code> is the index of the 
%mapping, 'property' is the name of the property which has to be changet and <CODE>mapvalue</CODE> is the new value. An even more convenient
%way to change mappings' fields is a achieved through the syntax s{n}.property = mapvalue.
%<dt>			params		 <dd>		sets the parameters to be optimised
%<dt>			dynamics		 <dd>		sets the dynamics of the system (please use add_dynamics instead)
%</dl>
%
%  @see get
%
%MAN_PAGE_END		

%this is a function that sets all the values. It actually calls the set
%of the parent class, so not particular check are necessary. It is
%important to do all the chack once the update process is finished, since
%during the update of an object temporary inconsistent states may
%arise. 

% Check that we have the right number of inputs
ni = nargin;
if (ni<3)|(ni/2 == floor(ni/2)),
  error('Wrong number of inputs'); 
end

if ~isa(m,'system'),
  % Call built-in SET. Handles calls like set(gcf,'user',system)	
  builtin('set',m,varargin{:});
  return
end

% For each couple of inputs
for i=1:(ni - 1)/2,
  name = varargin{i*2-1};
  value = varargin{i*2};
  
  if isstr(name),
    switch lower(name),
     case 'dynamics'
      error('use ''add_dynamics'' for setting the dynamics of the system'); ...
     case 'links'
      m.system.links=value;
     otherwise
      m.system = set(m.system,name,value);
    end
  else
    error('Arguments mismatch');
  end
end

if nargout == 0,
  assignin('caller',inputname(1),m)
end

