function m=add_rules(m,n,model_code,mode)
%ADD_RULES add specified number of rules
%    
%    This function is used to  specify the number and shape of
%    the rules associated to the TS model. 
%    
%    ADD_RULES(M,N)	 add N rules to the model M.
%    
%    ADD_RULES(M,N,MODEL_CODE)   add N rules to the model M,
%    MODEL_CODE is  a cell array of two elements {x y} where x
%    is the T-norm used (only 'product' is currently
%    implemented) and y is defuzzification method
%    ('meancentroid').
%    
%    ADD_RULES(M,N,MODEL_CODE,MODE)	 mode can take the value
%    'standard' or 'rulegrid'. In lhe latter case, the
%    position of the rules is initialised in order to fill the
%    space according to the position of the fuzzy sets along
%    the the different dimensions.
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose  add specified number of rules. This function is used to 
%   specify the number and shape of the rules associated to the
%   TS model. 
%
%   @synopsis add_rules(m,n)  
%   @description add <code>n</code> rules to the model <code>m</code>.
%
%   @synopsis add_rules(m,n,model_code)  
%   @description  add <CODE>n</CODE> rules to the model <CODE>m</CODE>, <CODE>model_code</CODE> is 
%   a cell array of two elements {x y} where x is the T-norm used (only 'product' is currently implemented)
%   and y is defuzzification method ('meancentroid').
%
%   @synopsis add_rules(m,n,model_code,mode)  
%   @description  mode can take the value 'standard' or 'rulegrid'. In lhe latter
%	case, the position of the rules is initialised in order to fill the space according
%	to the position of the fuzzy sets along the the different dimensions.
%
%MAN_PAGE_END  

if nargin<3,
	model_code = {'product' 'meancentroid'};
end

if nargin<4,
	mode = 'standard';
end

error(checkargs({m n model_code mode},[2 4],'mamdani','numeric','cell','char'));

%check for consistency
check(m);

and_code=model_code{1};
defuzz_code=model_code{2};

%check the inputs of the system
if get(m,'n_in')==0 & get(m,'n_out')==0
  error('number of inputs and number of outputs of the model have not been specified yet');
elseif get(m,'n_in')==0
  error('number of inputs of the model has not been specified yet');
elseif get(m,'n_out')==0
  error('number of outputs of the model has not been specified yet');
end

if isempty(m.model_code),
	m.model_code = model_code;
end

%First check if the rules we would like to add are compatible with the previous ones
if (strcmpi(m.model_code{1}, model_code{1}) & ...
	strcmpi(m.model_code{2}, model_code{2})),
	
	in_dim = get(m,'n_in');
	out_dim = get(m,'n_out');
	
	switch lower(mode),
	
	case 'standard',
	if (n>0),
		m.rls = [m.rls; ones(n,in_dim+out_dim)];
		
		for i=1:in_dim+out_dim,
			if isempty(m.mfs{i})
				m.mfs{i} = [1 0 0 0 0];
			end
		end
		m.n_rules = m.n_rules + n;

	end
	
	case 'rulegrid',
	
	nt = 1;
	for i=1:in_dim,	
		nt = nt * size(m.mfs{i},1);
	end
	
	if nt ~= 0,
		if n == -1,
			n = nt;
		end
		
		ref = cell(in_dim,1);
		lls = ref;
		
		for i=1:in_dim,
			lls{i} = 1:size(m.mfs{i},1);
		end
		
		[ref{:}] = ndgrid(lls{:});
		
		nref = reshape(cat(in_dim+1,ref{:}),[nt in_dim]);
		
		m.rls = [m.rls; [nref(1:n,:) zeros(n,out_dim)]];
			
		m.n_rules = m.n_rules + n;

	else
		error(['Can define a grid of rules only if fuzzy sets are already' 13 ...
				'defined along every dimension. Please use add_sets first']);
	end
	
	otherwise,
	end
	
else
	error('You cannot add rules of this type to the current model. Remove first the previous ones');
end





