function d=get(m,label,opt)
%GET gets the value of the attribute of an object
%    
%    D = GET(M,LABEL,OPTIONS) gets the value of the attribute
%    LABEL associated to the object M and return it inside D.
%    The following codes are recognised:
%    
%      NAME: returns the name of the object
%      N_IN: returns the number of inputs of the system
%      N_OUT: returns the number of outputs of the system
%      DATA: returns the data set linked to the system
%      T: returns the sampling period of the discrete
%          system, 0 if continuous
%      MAPPING: returns the array of mappings
%      LINKS: returns the links array which defines which
%          output is computed by which mapping
%      LIMITS: returns the bounds of the inputs and outputs
%      USERDATA: returns the 'userData' field
%      OPT: returns the 'opt' field
%      DATE: returns the date of creation of the object
%      MAPFIELD: let the user access directly property of
%          the mappings. When used, opt must contain a 2
%          elements cell array {N 'property'}. N is the
%          index of the  mapping and 'property' is the name
%          of the property which has to be accessed. An even
%          more convenient way to access mappings' fields is
%          a achieved through the syntax s{n}.property.
%      NUMPARAMS: returns the number of parameters to be
%          optimised
%      PARAMS: returns the parameters to be optimised
%      DYNAMICS: returns the dynamical description of the
%          regressors
%      HORIZ_LENGTH: returns the width of the time window
%          used to compute the outputs
%    
%    See also: SET
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose  gets the value of the attribute of an object.
%
%   @synopsis	d = get(m,label,options)
%   @description gets the value of the attribute <code>label</code>
%   associated to the object <code>m</code> and return it inside <code>d</code>.
%   The following codes are recognised:
%<dl>
%<dt>		name  <dd>	returns the name of the object
%<dt>     n_in  <dd>      	returns the number of inputs of the system
%<dt>     n_out  <dd>     	returns the number of outputs of the system
%<dt>			data <dd>		returns the data set linked to the system
%<dt>			T <dd>		returns the sampling period of the discrete system, 0 if continuous
%<dt>			mapping <dd>		returns the array of mappings
%<dt>			links <dd>			returns the links array which defines which output is computed by which mapping
%<dt>			limits <dd>			returns the bounds of the inputs and outputs
%<dt>			userData <dd>		returns the 'userData' field
%<dt>			opt			 <dd>		returns the 'opt' field
%<dt>			date		 <dd>		returns the date of creation of the object
%<dt>	 mapfield	 <dd>	let the user access directly property of the mappings. When used, opt
%must contain a 2 elements cell array {<code>n</code> 'property'}. <code>n</code> is the index of the 
%mapping and 'property' is the name of the property which has to be accessed. An even more convenient
%way to access mappings' fields is a achieved through the syntax s{n}.property.
%<dt>			numparams		 <dd>		returns the number of parameters to be optimised
%<dt>			params		 <dd>		returns the parameters to be optimised
%<dt>			dynamics		 <dd>		returns the dynamical description of the regressors
%<dt>			horiz_length		 <dd>		returns the width of the time window used to compute the outputs
%</dl>
%
%   @see set
%
%MAN_PAGE_END

%this function dies not perform any action on the external object. Just
%check for the consistency  of the input arguments.  
  
error(checkargs({m label},2,'external','char'));
  
switch lower(label),  
  %
  %	dynamics property
  %	-------------		
 case 'dynamics',
  d = m.dynamics;  

 case 'horiz_length', 
  %check the description of the dynamics
  if isempty(m.dynamics),
    error('the dynamics of the system has not been yet defined');
  end
  d=max(max(max(m.dynamics.ny+(m.dynamics.ny>0))),...
	max(max(m.dynamics.nu+m.dynamics.nd))); 
  
 otherwise,
  if nargin == 2,
    d = get(m.system,label);
  else
    d = get(m.system,label,opt);
  end
end


