function d=get(m,label,opt)
%GET gets the value of the attribute of an object
%    
%    D = GET(M,LABEL,OPT) gets the value of the attribute
%    LABEL associated to the object M and return it inside D.
%    The following codes are recognised:
%    
%      NAME: returns the name of the object
%      N_IN: returns the number of inputs of the system
%      N_OUT: returns the number of outputs of the system
%      DATA: returns the data set linked to the system
%      T: returns the sampling period of the discrete
%          system, 0 if continuous
%      MAPPING: returns the array of mappings
%      LINKS: returns the links array which defines which
%          output is computed by which mapping
%      LIMITS: returns the bounds of the inputs and outputs
%      USERDATA: returns the 'userData' field
%      OPT: returns the 'opt' field
%      DATE: returns the date of creation of the object
%      MAPFIELD: let the user access directly property of
%          the mappings. When used, opt must contain a 2
%          elements cell array {N 'property'}. N is the
%          index of the  mapping and 'property' is the name
%          of the property which has to be accessed. An even
%          more convenient way to access mappings' fields is
%          a achieved through the syntax s{n}.property. In
%          both syntax, if N is negative, the
%          <CODE>abs(n)</CODE> mapping describing the state
%          is accessed.
%      NUMPARAMS: returns the number of parameters to be
%          optimised
%      PARAMS: returns the parameters to be optimised
%      N_STATE: returns the number of states
%      STATE: returns the state vector of the system
%      STATEMAPPING: returns the mappings used to describe
%          the state
%      STATELINKS: returns the links array which defines
%          which state is computed using which statemapping
%      STATELIMITS: returns the bounds of the states
%      STATEMAPFIELD: let the user access directly property
%          of the statemappings. When used, opt must contain
%          a 2 elements cell array {N 'property'}. N is the
%          index of the  mapping and 'property' is the name
%          of the property which has to be accessed.
%    
%    See also: SET
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose  gets the value of the attribute of an object.
%
%   @synopsis	d = get(m,label,opt)
%   @description gets the value of the attribute <code>label</code>
%   associated to the object <code>m</code> and return it inside <code>d</code>.
%	The following codes are recognised:
%<dl>
%<dt>		name  <dd>	returns the name of the object
%<dt>     n_in  <dd>      	returns the number of inputs of the system
%<dt>     n_out  <dd>     	returns the number of outputs of the system
%<dt>			data <dd>		returns the data set linked to the system
%<dt>			T <dd>		returns the sampling period of the discrete system, 0 if continuous
%<dt>			mapping <dd>		returns the array of mappings
%<dt>			links <dd>			returns the links array which defines which output is computed by which mapping
%<dt>			limits <dd>			returns the bounds of the inputs and outputs
%<dt>			userData <dd>		returns the 'userData' field
%<dt>			opt			 <dd>		returns the 'opt' field
%<dt>			date		 <dd>		returns the date of creation of the object
%<dt>	 mapfield	 <dd>	let the user access directly property of the mappings. When used, opt
%must contain a 2 elements cell array {<code>n</code> 'property'}. <code>n</code> is the index of the 
%mapping and 'property' is the name of the property which has to be accessed. An even more convenient
%way to access mappings' fields is a achieved through the syntax s{n}.property. In both syntax,
%if <CODE>n</CODE> is negative, the <CODE>abs(n)</CODE> mapping describing the state is accessed.
%<dt>			numparams		 <dd>		returns the number of parameters to be optimised
%<dt>			params		 <dd>		returns the parameters to be optimised
%<dt>			n_state <dd>		returns the number of states
%<dt>			state <dd>			returns the state vector of the system
%<dt>			statemapping <dd>		returns the mappings used to describe the state
%<dt>			statelinks <dd>			returns the links array which defines which state is computed using which statemapping
%<dt>			statelimits <dd>			returns the bounds of the states
%<dt>	 statemapfield	 <dd>	let the user access directly property of the statemappings. When used, opt
%must contain a 2 elements cell array {<code>n</code> 'property'}. <code>n</code> is the index of the 
%mapping and 'property' is the name of the property which has to be accessed.
%
%</dl>
%
%   @see set
%
%MAN_PAGE_END   

error(checkargs({m label},2,'system','char'));

switch lower(label),
	
%
%	n_state property
%	-------------		
case 'n_state',
	d = m.n_state;

%
%	n_state property
%	-------------		
case 'state',
	d = m.state;

%
%	statemapping property
%	-------------		
case 'statemapping',
	d = m.statemapping;

%
%	statelinks property
%	-------------		
case 'statelinks',
	d = m.statelinks;

%
%	statelimits property
%	-------------		
case 'statelimits',
	d = m.statelimits;
	
%
%	other functionalities
%	---------------------
case 'mapfield',
	error(checkargs(opt, 2, 'numeric', 'char'));
	if opt{1}>0,
		d = get(m.system.mapping{opt{1}}, opt{2});
	else
		d = get(m.statemapping{-opt{1}}, opt{2});
	end
		
otherwise,
	if nargin == 2,
		d = get(m.system,label);
	else
		d = get(m.system,label,opt);
	end
		
end
	
		
