function [ClosestCone]=conse_cc(Inputs,Outputs,Environment)

%CONSE_CC
%
%   CLOSESTCONE=CONSE_CC(INPUTS,OUTPUTS) calculates the closest linear
%   controller to the controller defined by I/O pairs given in INPUTS
%   and OUTPUTS.
%
%   The closest linear controller to H(x) is the matrix Cp which provides
%   the lowest square error e^2 = sum (H(x)-Cpx)^2.
%
%   CLOSESTCONE=CONSE_CC(INPUTS,OUTPUTS,ENVIRONMENT) calculates the closest
%   linear controller to the controller defined by I/O pairs given in INPUTS
%   and OUTPUTS in the the origin ENVIRONMENT.
%
%   The closest linear controller to H(x) in a given origin environment is the
%   matrix Cp which provides the lowest square error e^2 = sum (H(x)-Cpx)^2 in
%   such environment.
%
%   If ENVIRONMENT is scalar then an input x belongs to the origin environment
%   when |x| < ENVIRONMENT.
%
%   If ENVIRONMENT is a column vector with so elements as rows has INPUTS, then
%   an input x belongs to the origin environment when |x(i)| < ENVIRONMENT(i)
%   for all x(i).
%
%   See also conse_gr, conse_co, conse_te, conse_ga, coneval, inabcd, insfunc
%


%-----------------------------------------------
%  Departamento de Sistemas y Automatica
%  Universidad de Sevilla
%-----------------------------------------------
%  File:  conse_cc.m
%  Project:  FAMIMO
%  Block:  Conicity
%  Use:  Master function
%-----------------------------------------------
%  Toolbox used:   
%  Other functions used: 
%-----------------------------------------------
%  Authors:  Javier Martin Recio  
%  E-mail:  jmartin@cartuja.us.es
%  Date: Sep/30th/97
%-----------------------------------------------
%  Last update
%   Date:  
%   Name:  
%   Main changes:
%
%   Version:  1.0
%-----------------------------------------------
%  System:  MATLAB 4.2b or higher
%-----------------------------------------------
%  If you have any question please contact us: 
%  E-mail ...
%-----------------------------------------------

%-------------------------------------------------------------------------
%  Main variables description: 
%
%   vRem : Vector that indicates the inputs and outputs to be removed
%       because they hkare outside the origin environment.
%   mRem : Auxiliary matrix for inputs and output removing.
%   mInputs : Inputs that belongs to origin environment
%   mOutputs : Outputs that belongs to origin environment
%
%-------------------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Checking input arguments %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

error(nargchk(2,3,nargin));

nOut=size(Outputs,1);           % Number of outputs
[nIn,nInVal]=size(Inputs);      % Number of inputs and number of input values

if nInVal~=size(Outputs,2)
  error(' Number of INPUTS and OUTPUTS columns must agree');
end;


if nargin==3
%% External inputs do not have to be considered.

  lengIn=length(Environment);

  if lengIn~=1 & lengIn~=nIn
      error(' Bad number of elements in parameter ENVIRONMENT')
  end;

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % External inputs removing %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% Environment based in input norm
   if lengIn==1,
     vRem=sum([Inputs.^2;zeros(1,nInVal)])<Environment^2;   % |x|^2<Environment^2
 
  %% Environment based in each input variable norm
   else
     vAux=abs(Inputs)<Environment*ones(1,nInVal);       % |x(i)|<Environment(i)
     vRem=sum([vAux;zeros(1,nInVal)])>0;
   end;

  %% External inputs removing
   mRem=ones(nIn,1)*vRem;
   mInputs=Inputs.*mRem;    % Inputs removing
   mRem=ones(nOut,1)*vRem;
   mOutputs=Outputs.*mRem;  % Outputs removing

else
%% Consider all inputs

 mInputs=Inputs;
 mOutputs=Outputs;

end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Closest linear and static system %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

ClosestCone=mOutputs/mInputs;

%%%%%%%%%%%%%%%%%%%%%%%%
% End of this function %
%%%%%%%%%%%%%%%%%%%%%%%%
